/*****************************************************************
    Filename: MIP_EXEC.C
    Copyright 1992, 1993 (C) Echelon Corp. All rights reserved
    %W% %G% %U%
    Modified 10/10/94	By Don M. Scofield and Birkir Brimdal
			For LM_104
    Modified 10/16/94   by MHS for DMS
    Modified 11/03/94   by BB
    Modified 12/09/94   by BB
    Modified 04/04/97   by DMS updated the version to 2.00 and the
			copyright to 1997. No other changes in
			this file. (MIP_PIO.C was changed)
    Modified 04/22/97   by BB updated the version to 2.10 and 
                        changed command line parsing to allow
                        any IRQ number from 3 to 14
*****************************************************************/

/*
    This module may be compiled to support one of three MIP interfaces:

    1) The basic polled I/O model. In this case, no interrupts are
    provided by the MIP. Background polling of the MIP occurs under
    the control of the DOS TICK interrupt, every 55ms. Polling may also
    occur whenever the read() or write() functions are called. 'Polling'
    consists of writing downlink and waiting for an uplink response,
    which may be a data transfer or a NULL token transfer. This version
    will work with both the MIP P20 and MIP P50 applications. A driver
    option may be set forcing the polling function to wait for the token
    before returning.

    2) Interrupt driven model. This model works with the MIP P50 only.
    In this model the Host (this driver) normally keeps the parallel
    I/O 'token'. Downlink transfers occur as needed. Uplink transfers
    occur via two interrupts from the MIP hardware. The first interrupt
    is a signal that the MIP does not own the token, and needs to send
    data uplink. The driver writes downlink, giving up the token. The
    second interrupt is a signal the the MIP is starting an uplink
    transfer. In this case the driver reads the uplink data from the
    MIP. Both interrupts appear on a single IRQ since they are generated
    in the same manner by the MIP. The distiction between them is made
    by evaluating the token state of the driver. No 'polling' occurs
    in this model, it is completley interrupt driven in the uplink
    transfer case.

    3) Interrupt + DMA model. Operation of this model is the same as
    for the second model, except that one of the PC/AT's DMA channels
    is used to perform the data transfers. A separate DMA END interrupt
    is provided to signal the end of the DMA transfer. Note that the first,
    and sometimes second, bytes must be read/written using polled I/O
    since the transfer type and length need to be examined before
    launching the DMA transfer. Depending on the machine this driver
    is being used on, this may or may not be a faster design than
    model (2) since DMA setup requires a fair amount of work.

    The UPLINT definition enables model (2). UPLINT + DMAENA enables
    model (3).

    The number of Interface States defined are overkill for the simplist
    model, model (1). These states become more important for models (2)
    and (3).

    The include files for this product give a full description of the
    hardware properties of the MIP interface which this driver uses.

*/
/****************************************************************/
//#define UPLINT			// enable uplink interrupt proc.
#ifdef UPLINT
//#define DMAENA		// enables the DMA interface
#endif
//#define DBO			// enables LPT1 state debugging
/****************************************************************/

#define EXT
/* The segment, class, and group names prescribed in this module
** cause this entire module to be in the same physical segment.
** ALL modules used in this driver must use a similar segment
** control. */

// This sets the data and BSS class to 'CODE':
#pragma option -zTCODE -zBCODE

// This sets the code, BSS, and data group to 'DGROUP':
#pragma option -zPDGROUP -zSDGROUP -zGDGROUP


#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>

#include "mip_typs.h"
#include "mip_drvr.h"
#include "mdv_time.h"

// State debugging:
#ifdef DBO
#define DBSO	outportb(0x3BC, ifc_state + (mip_state << 3))
#define DBEO	outportb(0x3BC, 0xFF)
#define DBIO	outportb(0x3BE, 0)

#else
#define DBSO
#define DBEO
#define DBIO
#endif

#ifdef UPLINT
unsigned int mip_intrq; 	// IRQ number for this hardware.
static unsigned int pic_base;
void interrupt mip_isr(void);
#else
extern void interrupt tick_int_main(void);
void mip_isr(BOOLEAN from_int, BOOLEAN token_wait);
#endif

extern void prnt(char far *sp);
extern void dosptc(char cc);
extern void dosout_int(unsigned char i);

/* extern boolean debug; */
extern BOOLEAN debug;

// micro_delay
extern unsigned int micro_delay;       

#ifdef DMAENA
// Allowable DMA channels for this implementation are 0-3. These
// are the 8 bit bus DMA channels. Channels 4-7 must always
// transfer an even number of bytes, and are not applicable to this
// MIP without modifications.
static unsigned char dma_channel;	// 0-3
// I/O register addresses for the DMA controller.
static unsigned int dma_maskreg;
static unsigned int dma_pagereg;
static unsigned int dma_basereg;
#endif

// Driver globals
int ldv_handle			= -1;
BOOLEAN ldv_open		= FALSE;
int driver_busy			= 0;
unsigned down_cmd_local 	= miNULL;
BOOLEAN mip_wants_token 	= FALSE;
BOOLEAN callback_service 	= FALSE;
BOOLEAN source_quenched		= FALSE;
BOOLEAN g_afc_disable		= FALSE;
// Driver+MIP states:
IState ifc_state 		= IStateIdle1;
MipState mip_state 		= MIP_IDLE;



// Debugging counters:
int dbg_upl_tossed		= 0;
int dbg_xoffed			= 0;
int dbg_istate_unks		= 0;

// This is the user installed read callback function pointer.
static void interrupt (*callback_read)(void) = NULL;
// This is either the old int. vector, or the DOS_TICK vector.
void interrupt (*vector_was)(void);


// These are the driver's input/output buffer pointers
PioXferWhole *pxi_head, *pxi_tail;	// head/tail for input buffers
PioXferWhole *pxo_head, *pxo_tail;	// head/tail for output buffers
PioXferWhole *px_local;			// single local buffer

// default buffer counts:
int pxo_count = 8;
int pxi_count = 8;

// The DMA controller or PIO transfer pointer:
PioXferWhole *xfer_ptr;

// This is linked at the end of the driver code/data area:
// Configurable buffers are built starting at this location:
extern unsigned int Driver_len;
// This structure contains the driver device name string: "LON[n]"
extern struct dev_header_s Header;

// Local prototypes:
void callback_process(void);

/****************************************************************/
/* DRIVER BUFFER INITIALIZATION                                 */
/****************************************************************/

/*
** Initialization utility for pmip_driver_init() Build up a list
** of driver buffers.
** ! DMA version must check for 64k page boundary violations.
*/
#pragma warn -def
PioXferWhole *init_queue(int q_count, PioXferWhole **pxbase) {
    PioXferWhole *pxp;
    PioXferWhole *last_pxp;
    BOOLEAN first = TRUE;
    union {
	unsigned long l_addr;
	unsigned char b_addr[4];
	unsigned int w_addr[2];
    } addr1, addr2;

    /* Init the forward list links & busy flag */
    pxp = *pxbase;
    while(q_count--) {
	// For the DMA case, check for page violations.
	addr1.l_addr = FP_OFF(pxp) + ((long)FP_SEG(pxp) << 4);
	addr2.l_addr = FP_OFF(pxp+1) + ((long)FP_SEG(pxp+1) << 4);
	if(addr1.w_addr[1] != addr2.w_addr[1]) {
	    // page violation, skip this buffer.
	    pxp++;
	    if(first)
		*pxbase = pxp;
	    else
		last_pxp->next = pxp;
	}
	first = FALSE;

	memset(pxp, 0, sizeof(PioXferWhole));
	pxp->next = (q_count) ? pxp+1 : *pxbase;
	pxp->busy = FALSE;
	pxp->cmd_xfer = CMD_XFER;
	last_pxp = pxp;
	pxp++;
    }
    return(pxp);
}
#pragma warn +def

/*
** This is invoked upon startup of this driver, and whenever the device
** is OPENed. It builds the run time buffer space starting at Driver_len.
** Return end-of-driver address.
**
** Assume <pxo_count> and <pxi_count> have been set.
*/
byte *drv_buffers_init(void) {
    PioXferWhole *end_offs;

    pxi_head = (PioXferWhole *) &Driver_len;
    // Init the input buffer list.
    pxo_head = init_queue(pxi_count, &pxi_head);
    pxi_tail = pxi_head;

    // Init the output buffer list.
    px_local = init_queue(pxo_count, &pxo_head);
    pxo_tail = pxo_head;

    // Init the local buffer list of one.
    end_offs = init_queue(1, &px_local);

    mip_state = MIP_IDLE;

    return((byte *)end_offs);
}

/****************************************************************/
/* HARDWARE INTERFACE FUNCTIONS                                 */
/****************************************************************/

#ifdef UPLINT
/*
** This function, our_setvect(), has been created due to a crash scenario
** which occurs if the client application calls close() from exit() code.
** The close() function winds up here and in turn, if the system setvect()
** function is used, goes back into DOS and fries. Restoration of this
** interrupt vector is optional anyway.
*/
static void our_setvect(unsigned int_no, void interrupt (*isr)()) {

    asm xor	ax, ax
    asm mov	es, ax
    _BX = int_no;
    asm	shl	bx, 1
    asm	shl	bx, 1
    asm	pushf
    asm	cli
    _AX = FP_OFF(isr);
    asm	mov	es:[bx], ax
    _AX = FP_SEG(isr);
    asm	mov	es:[bx]+2, ax
    asm	popf

}

/*
** Mask/Unmask the uplink/DMAend interrupt.
*/
void upl_imask(BOOLEAN if_mask) {
    int imask;

    if(mip_intrq < 8)
	imask = (1 << mip_intrq);
    else
	imask = (1 << (mip_intrq-8));
    disable();
    // Mask/Un-mask the interrupt:
    if(if_mask)
	outportb(pic_base+IMR_OFFS, inportb(pic_base+IMR_OFFS) | imask);
    else
	outportb(pic_base+IMR_OFFS, inportb(pic_base+IMR_OFFS) & (~imask));
    enable();
}
#else
#define upl_imask(x)
#endif

/*
** Sync up with the MIP after resets. Return TRUE if complete.
*/
static BOOLEAN sync_master(BOOLEAN if_wait) {
    byte i_data;
    static LdvTimer l_timer;

    // Wait up to 4 seconds for HS:
    timer_start(4000, &l_timer);
    while(read_hs()) {
	if(!if_wait)
	    return(FALSE);
	if(timeout_check(&l_timer))
	    return(FALSE);
    }

    /* Output the CMD_RESYNC on the bus. */
    pio_write(CMD_RESYNC);

    /* Now the EOM */
    pio_write(0);

    /* Look for the second sync byte, CMD_ACKSYNC, from the NEURON. */
    i_data = pio_read();
    return(i_data == CMD_ACKSYNC);
}

/*
** Use this service whenever the MIP has reset.
** Use <if_wait> to determine whether to wait around for the MIP to
** come out of reset, or just back off for now since this process
** takes time.
*/
int mip_reset_service(BOOLEAN if_wait) {
    mip_was_reset = FALSE;
    mip_state = MIP_IDLE;
    ifc_state = IStateIdle1;
    DBEO;

    // Sync up with the MIP's PIO interface.
    if(!sync_master(if_wait)) {
	mip_was_reset = TRUE;
	return(DRV_NOT_READY);
    }

    mip_was_reset = FALSE;
    DBSO;
    return(DRV_OKAY);
}

#ifdef UPLINT
/*
** This function sets up some DMA port addresses, IRQ numbers,
** and sets up the interrupt.
*/
void hardware_init(BOOLEAN if_setup) {
    void interrupt (*vect_temp)(void);
    static int vector_n;

#ifdef DMAENA
    static unsigned char page_table[] = {
    // table for page register addresses
     0x87, 0x83, 0x81, 0x82);,	// for dma channels 0, 1, 2, 3
#endif

    if(!if_setup) {
	// called on CLOSE of device.
	// Mask out the MIP/DMA interrupts.
	upl_imask(TRUE);
	// Put back the int vector
	our_setvect(vector_n, vector_was);
#ifdef DMAENA
	outportb(dma_maskreg, DMA_MASK | dma_channel);
#endif
	return;
    }

    // Set up <pic_base>, vector_n here:
    if(mip_intrq < 8) {
	pic_base = PIC_1;
	vector_n = 8 + mip_intrq;
    } else {
	pic_base = PIC_2;
	vector_n = 0x70 - 8 + mip_intrq;
    }
    upl_imask(TRUE);

#ifdef DMAENA
    dma_pagereg = page_table[dma_channel];
    dma_basereg = DMA0_BASE + dma_channel * 2;
    dma_maskreg = DMA0_BASE + 10;
    // shut down the DMA hardware at this time.
    outportb(dma_maskreg, DMA_MASK | dma_channel);
#endif
    // Install the interrupt vector.
    vector_was = getvect(vector_n);
    vect_temp = MK_FP(_CS, FP_OFF(mip_isr));
    setvect(vector_n, vect_temp);
    // set things right with a specific EOI:
    outportb(pic_base+EOI_OFFS, EOI_CODE + (mip_intrq & 0x07));
}
#else
#define hardware_init(x)
#endif

/****************************************************************/
/* PARALLEL I/O TRANSFER FUNCTIONS                              */
/****************************************************************/

/*
** This function is used to handle the end of all transfers (up/down link),
** whether by DMA or polled I/O. Mostly it deals with tidying up at the
** end of the transfer and freeing any resources.
*/
void end_xfer_process(void) {
    switch(ifc_state) {
    case IStateDownXfer:
#ifndef UPLINT
	// Check for downlink reset commands. This is not required
	// when the hardware reset sense latch works properly.
	if(xfer_ptr->mip_cmd == miRESET)
	    mip_was_reset = TRUE;
#endif

	// free the output buffer if needed.
	if(mip_state != MIP_ACK_WAIT) {
	    if(xfer_ptr == pxo_tail) {
		pxo_tail->busy = FALSE;
		pxo_tail = pxo_tail->next;
		mip_state = MIP_IDLE;
	    }
	}
	// fall into:
    case IStateDownNull:

	ifc_state = IStateIdle2;
	break;

    case IStateUpXferB:

	// Process the input transfer:
	if(mip_state == MIP_ACK_WAIT &&
	 xfer_ptr->mip_cmd == miACK) {
	    mip_state = MIP_ACK_OKAY;
	} else {	// Do not post miACKs unless error..
	    if(xfer_ptr == pxi_head) {
		pxi_head->busy = TRUE;
		pxi_head = pxi_head->next;
	    }
	    // Driver must test for the miRESET command and
	    // use down_cmd_local to post a miFLUSH_CANCEL.
	    if(xfer_ptr->mip_cmd == miRESET && !g_afc_disable)
		down_cmd_local = miFLUSH_CANCEL;
	    // Always check for the need for a source quench.
	    else if(pxi_head->busy == TRUE && !source_quenched)
		down_cmd_local = miPUPXOFF;
	    callback_service = TRUE;
	}
	ifc_state = IStateIdle1;
	break;


    // IStateIdle1, IStateIdle2, IStateUpXferA no action.
    default:
	dbg_istate_unks++;
	break;

    } // end of switch.
    DBSO;
}


/*
** Start a DMA read/write operation, OR use polled I/O.
** <dma_mode> is either DMA_MODE_IN or DMA_MODE_OUT.
*/
void transfer_start(void *vptr, unsigned int count, unsigned char dma_mode,
 IState new_state) {
    register byte *bptr;
    register unsigned int r_count;
#ifdef DMAENA
    union {
	unsigned long l_addr;
	unsigned char b_addr[4];
	unsigned int w_addr[2];
    } addr;
#endif

    bptr = (byte *)vptr;
    r_count = count;
    ifc_state = new_state;
    DBSO;

    if(dma_mode == DMA_MODE_OUT) {
    // When doing a write to the PIO interface, include an extra NULL
    // at the end of the data which will form the EOM.
    // Also, our hardware requires an initial IN or OUT operation to be
    // executed by the CPU in order to set up the DMA direction hardware,
    // and to clear any lingering DMA requests.
    // Therefore the first byte is written under CPU control. This should
    // not be a problem since the HS signal is typically at 'GO' by this
    // time. If DMA_MODE_IN then the first byte has already been IN'd by
    // the caller.
	*(bptr+r_count) = 0;
	pio_write(*bptr++);
	// leave <r_count> bumped by one for the EOM.
	// Clear this since downlink transfers satisfy it's being set.
	mip_wants_token = FALSE;
	// If the downlink transfer is a NULL transfer, use of the DMA
	// hardware becomes a bit ludicrous. Uplink NULLs don't even get
	// to this point in the code.
	if(r_count == 1) {
	    pio_write(*bptr);
	    end_xfer_process();
	    return;
	}
    }
#ifdef DMAENA
    addr.l_addr = FP_OFF(bptr) + ((long)FP_SEG(bptr) << 4);
    // Always decrement <r_count> since terminal count is at 0xFFFF.
    r_count--;
    disable();
    // Due to the intense I/O goings on for DMA setup, the following
    // is written in assembly:
    // Mask out the DMA channel during setup:
    asm	mov	al, dma_channel
    asm or	al, DMA_MASK
    asm	out	DMA_MASK_OFS, al

    asm	mov	dx, dma_basereg
    // Out with the addresses:
    _AX = addr.w_addr[0];
    asm	out	dx, al
    asm	mov	al, ah
    asm	out	dx, al
    // Out with the 16 bit count:
    asm	inc	dx
    asm	mov	ax, r_count
    asm	out	dx, al
    asm	mov	al, ah
    asm	out	dx, al
    // And the page:
    outportb(dma_pagereg, addr.b_addr[2]);
    // And the new mode:
    asm	mov	al, dma_channel
    asm	add	al, dma_mode
    asm	out	DMA_MODE_OFS, al
    // De-maskify and go:
    asm	mov	al, dma_channel
    asm	out	DMA_MASK_OFS, al
    enable();
#else
    // Using polled I/O, read|write the data at the parallel port.
    if(dma_mode == DMA_MODE_OUT) {
	while(r_count--)
	    pio_write(*bptr++);
    } else {
	while(r_count--)
	    *bptr++ = pio_read();
    }
    end_xfer_process();
#endif
    DBSO;
}

/*
** This is the downlink check/kickoff function. It first checks the state
** of <ifc_state>, and if Idle1, will start the necessary downlink DMA
** transfer. <xfer_length> must reflect the length as passed to the parallel
** interface in the NEURON.
** Re-entrancy protection is provided to prevent this from being executed
** by the read/write functions and interrupts at the same time via <in_dlko>.
** Use this function for ALL downlink transfers.
*/
void downlink_kickoff(void) {
    int mip_cmd;
    BOOLEAN comm_down;
    static BOOLEAN in_dlko = FALSE;

    disable();
    if(in_dlko) {
	enable();
	return;
    }
    in_dlko = TRUE;
    enable();

    if(ifc_state == IStateIdle1) {

	if(down_cmd_local) {
	    px_local->xfer_length = 1;
	    px_local->cmd_xfer = CMD_XFER;
	    px_local->mip_cmd = down_cmd_local;
	    if(down_cmd_local == miPUPXOFF) {
		source_quenched = TRUE;
		dbg_xoffed++;
	    } else if(down_cmd_local == miPUPXON)
		source_quenched = FALSE;
	    down_cmd_local = miNULL;
	    xfer_ptr = px_local;
	    transfer_start(&px_local->cmd_xfer, 3, DMA_MODE_OUT,
	     IStateDownXfer);

	} else if(pxo_tail->busy) {
	    mip_cmd = pxo_tail->mip_cmd & miTYPE;
	    comm_down = (mip_cmd == miCOMM || mip_cmd == miNETMGMT);

	    if(comm_down && mip_state == MIP_IDLE) {
		// Turn this into a MIP output buffer request. We use
		// px_local since transfer_start() will stuff a NULL at
		// the 'end'.
		px_local->xfer_length = 1;
		px_local->cmd_xfer = CMD_XFER;
		px_local->mip_cmd = pxo_tail->mip_cmd;
		xfer_ptr = px_local;
		transfer_start(&px_local->cmd_xfer, 3, DMA_MODE_OUT,
		 IStateDownXfer);
		mip_state = MIP_ACK_WAIT;

	    } else if((comm_down && mip_state == MIP_ACK_OKAY) || !comm_down) {
		// Otherwise, send the buffer contents.
		xfer_ptr = pxo_tail;
		transfer_start(&pxo_tail->cmd_xfer, pxo_tail->xfer_length+2,
		 DMA_MODE_OUT, IStateDownXfer);
		mip_state = MIP_IDLE;
	    }
	}

	// If nothing has gone down and ...
	if(mip_wants_token && ifc_state == IStateIdle1) {
	    // Uplink interrupt was received during a busy ifc_state.
	    // Transfer the downlink NULL token.
	    px_local->cmd_xfer = CMD_NULL;
	    xfer_ptr = px_local;
	    transfer_start(&px_local->cmd_xfer, 1, DMA_MODE_OUT,
	     IStateDownNull);
	}

    }
    DBSO;
    in_dlko = FALSE;
}

/****************************************************************/
/* INTERRUPT SERVICE FUNCTIONS                                  */
/****************************************************************/

/*
** Sub function used by mip_isr().
** Actions are based on the <ifc_state>.
*/
void ul_int_service(void) {
    int pio_b1;

    // Uplink interrupt from MIP:
    switch(ifc_state) {
    case IStateIdle1:
	// MIP needs the token. Start a downlink transfer,
	// if nothing queued then send the NULL token.
	mip_wants_token = TRUE;
	downlink_kickoff();
	break;

    case IStateDownXfer:
    case IStateUpXferB:
	// MIP sent an uplink int just as we were starting
	// or ending a downlink xfer.
	mip_wants_token = TRUE;
	break;

    case IStateIdle2:
	// MIP owns the token and is starting an uplink xfer.
	// Using polled I/O, read the cmd byte and typically
	// the length byte. Then, if not a niNULL, start an
	// uplink xfer.
	ifc_state = IStateUpXferA;
	DBSO;
	if((pio_b1 = pio_read()) == CMD_XFER) {
	    pio_b1 = pio_read();	// read the length byte.
	    if(pio_b1) {

	    // All eight bit transfer lengths are valid. No range
	    // checking is required. Set up a buffer for reading the
	    // transfer. If there are no input buffers available use the
	    // local buffer.
		xfer_ptr = (pxi_head->busy) ? px_local : pxi_head;
		if(pxi_head->busy)
		    dbg_upl_tossed++;
		transfer_start(&xfer_ptr->mip_cmd, pio_b1,
		 DMA_MODE_IN, IStateUpXferB);
		xfer_ptr->xfer_length = pio_b1;
	    } else
		// Null transfer, we now own token.
		ifc_state = IStateIdle1;
	} else
	    // Null transfer, we now own token.
	    ifc_state = IStateIdle1;
	break;

    // IStateDownNull, IStateUpXferA no action.
    default:
	dbg_istate_unks++;
	break;

    }  // end of switch.
    DBSO;

}

#ifdef UPLINT

#ifdef DMAENA
#define ISR_TEST	(RISR_UL_INT|RISR_DMA_END|RISR_RESET)
#else
#define ISR_TEST        (RISR_UL_INT|RISR_RESET)
#endif
/*
** This is the DMA End / MIP Interrupt Service Routine. Actions are based
** on the current <ifc_state> and the interrupt source: MIP Uplink or
** DMA end.
** If a MIP reset is sensed, try and re-sync, but do not wait on the MIP.
*/
void interrupt mip_isr(void) {
    int irq;

    // Allow higher priority interrupts:
    enable();
    // The cascaded PIC needs two EOIs
    if(mip_intrq > 7)
	outportb(PIC_1+EOI_OFFS, NS_EOI);
    if(ldv_open) {

	while( (irq = read_isr()) & ISR_TEST )
	{
	    if(mip_was_reset)
		break;
	    if(irq & RISR_UL_INT)
	    {
		// Uplink interrupt from MIP:
		ul_int_service();
	    }

#ifdef DMAENA
	    if(irq & RISR_DMA_END)
		// DMA End interrupt:
		end_xfer_process();
#endif
	    // H.W. Indicates the MIP has reset.
	    if(irq & RISR_RESET)
	    {
		// Kablooie.
		mip_was_reset = TRUE;
	    }
	    DBSO;
	}
	// Start any required downlink transfers.
	if(!mip_was_reset)
	    downlink_kickoff();
	else
	    (void)mip_reset_service(FALSE);

	if(callback_service)
	    callback_process();
    }

    // Do the EOI:
    outportb(pic_base+EOI_OFFS, NS_EOI);
}

#else

/*
** This is the hooked DOS TICK interrupt function for the POLLED I/O
** version. This function may be called from either the TICK interrupt
** or from the read() or write() functions in order to process transfers
** at the MIP link-layer. When called from the TICK int, all registers
** have been saved, interrupts re-enabled, and DS=CS. The token_wait
** flag is used to force us to wait for an uplink transfer while in this
** function.
*/
void mip_isr(BOOLEAN from_int, BOOLEAN token_wait) {
    static LdvTimer l_timer;

    if(!ldv_open || (from_int && driver_busy))
	return;
    // Test h.w. for MIP reset indication.
    if(read_isr() & RISR_RESET)
	// Kablooie.
	mip_was_reset = TRUE;
    if(!mip_was_reset) {
	// If we own the token then send something downlink.
	if(ifc_state == IStateIdle1)
	    ul_int_service();
	// If this is an interrupt thread or read(), wait a bit for
	// uplink activity. Never wait more than 4ms.
	if(token_wait) {
	    timer_start(4, &l_timer);
	    while(!timeout_check(&l_timer))
		if(!read_hs())
		    break;
	}
	// Now process downlink requirements.
	if(!read_hs())
	    ul_int_service();
    } else
	(void)mip_reset_service(FALSE);
    if(callback_service)
	callback_process();
}

#endif


/****************************************************************/
/* HOST INTERFACE FUNCTIONS                                     */
/****************************************************************/

/*
** Test for and process a callback.
** Called when callback_service is TRUE.
*/
void callback_process(void) {
    static BOOLEAN in_callback = FALSE;

    if(callback_read && !in_callback && driver_busy == 0) {
	in_callback = TRUE;
	_AX = ldv_handle;
	(*callback_read)();
	in_callback = FALSE;
    }
    callback_service = FALSE;
}


/*
** Convert DOS return codes to API/LDV return codes.
*/
int ldv_code(int code) {
    if(code == DRV_OKAY)
	return(LDV_OK);
    if(code == DRV_UNKNOWN_UNIT)
	return(LDV_NOT_OPEN);
    if(code == DRV_WRITE_ERROR)
	return(LDV_NO_BUFF_AVAIL);
    if(code == DRV_READ_ERROR)
	return(LDV_NO_MSG_AVAIL);
    if(code == DRV_DEV_FAIL)
	return(LDV_DEVICE_ERR);
    if(code == DRV_NOT_READY)
	return(LDV_DEVICE_BUSY);
    return(code);
}


/*
** ldv_read_fn()
** This function is called whenever the DOS read to the LON(n) device is
** executed. No blocking occurs here.
*/
int ldv_read_fn(NetMsg far *msg_p, int len) {

    /* Verify that the device is open: */
    if(!ldv_open)
	return(DRV_UNKNOWN_UNIT);
    if(mip_was_reset) {
	if(mip_reset_service(FALSE) != DRV_OKAY)
	    return(DRV_NOT_READY);
    }
#ifndef UPLINT
    // Service the interface
    mip_isr(FALSE, TRUE);
#endif

    /* Check if a buffer needs servicing. */
    if(pxi_tail->busy) {
	/* Check xfer_length, this is the length of all but the
	** length field. Just chop the message if there is a
	** length problem. */
	if(len < pxi_tail->xfer_length+1)
		pxi_tail->xfer_length = len-1;
	/* move data starting at .xfer_length */
	_fmemcpy(msg_p, &pxi_tail->xfer_length, pxi_tail->xfer_length+1);
	/* swap these to match locations. */
	msg_p->xfer_length = pxi_tail->xfer_length;
	msg_p->mip_cmd = pxi_tail->mip_cmd;
	/* free the buffer */
	pxi_tail->busy = FALSE;
	pxi_tail = pxi_tail->next;
	/* Check for XON requirement. */
	if(source_quenched && pxi_head->next->busy == FALSE) {
	    down_cmd_local = miPUPXON;
	    downlink_kickoff();
	}

	return(LDV_OK);
    }
    /* Else, if no messages are buffered, return a length 0. */
    msg_p->xfer_length = 0;
    return(LDV_OK);
}


/*
** ldv_write_fn()
** This function is called whenever the DOS write to LON(n) is
** executed.
*/
int ldv_write_fn(NetMsg far *msg_p, int len) {

    // Verify that the device is open:
    if(!ldv_open)
	return(DRV_UNKNOWN_UNIT);
    if(mip_was_reset) {
	if(mip_reset_service(FALSE) != DRV_OKAY)
	    return(DRV_NOT_READY);
    }
    // Check for a downlink miRESET command: This can zing down regardless
    // of any buffered messages.
    if(msg_p->mip_cmd == miRESET) {
	(void)drv_buffers_init();
	down_cmd_local = miRESET;
    } else {

	// check for a free buffer.
	if(pxo_head->busy)
	    return(DRV_WRITE_ERROR);

	if(len > 2 + PIO_DATA_MAX)
	    return(DRV_WRITE_ERROR);	// hey, data size exceeded.
	/* move data starting at .xfer_length
	** msg_p->xfer_length may be zero, for cmd only messages. */
	_fmemcpy(&pxo_head->xfer_length, msg_p, len);
	/* swap these to match locations. */
	pxo_head->xfer_length = msg_p->xfer_length+1;
	pxo_head->mip_cmd = msg_p->mip_cmd;
	/* post this buffer for output */
	pxo_head->busy = TRUE;
	pxo_head = pxo_head->next;
    }
#ifndef UPLINT
    // Service the interface
    mip_isr(FALSE, FALSE);
#else
    // Try and start a downlink transfer.
    downlink_kickoff();
#endif
    return(LDV_OK);
}


/*
** ldv_write_direct()
** This function is called directly by the client rather that via DOS.
*/
int far _loadds ldv_write_direct(NetMsg far *msg_p, int len) {
    int rval;

    /* Prevent re-entrancy of the write function. */
    if(driver_busy)
	return(LDV_NO_BUFF_AVAIL);

    // this is assumed to be an atomic op:
    driver_busy++;
    rval = ldv_write_fn(msg_p, len);
    driver_busy--;
    return(ldv_code(rval));
}


/*
** ldv_read_direct()
** This function is called directly by the client rather that via DOS.
** In the case of when a read is attempted when there is no input
** ready this function returns an error status.
*/
int far _loadds ldv_read_direct(NetMsg far *msg_p, int len) {
    int rval;

    // this is assumed to be an atomic op:
    driver_busy++;
    rval = ldv_read_fn(msg_p, len);
    driver_busy--;

    if(rval == 0 && msg_p->xfer_length == 0)
	return(LDV_NO_MSG_AVAIL);
    /* else decrement the length field to exclude the command byte. */
    if(msg_p->xfer_length)
	msg_p->xfer_length--;
    return(ldv_code(rval));
}


/*
** OPEN or CLOSE the device. If close, just clear the callback function
** and handle. If open, reset the MIP and sync up with it.
*/
int ldv_open_close_fn(BOOLEAN if_open) {

    if(if_open) {
	if(ldv_open)
	    return(LDV_ALREADY_OPEN);
	timer_mode_init();
	hardware_init(TRUE);
	reset_mip();

	// Always init the buffers and states
	(void) drv_buffers_init();
	if(mip_reset_service(TRUE) != DRV_OKAY)
	    return(DRV_NOT_READY);
	ldv_open = TRUE;
	// Enable uplink interrupts.
	upl_imask(FALSE);
	DBIO;
    } else {		// Close the device.
	hardware_init(FALSE);
	ldv_open = FALSE;
	callback_read = NULL;
	ldv_handle = -1;
    }
    return(DRV_OKAY);
}


/*
** This function is called directly by the driver client to register
** the 'callback' function and set the handle. This function's address
** was returned as part of the IOCTL function. A FAR return must be executed
** by the callback function.
*/
void far _loadds ldv_register_fn(int handle,
	void interrupt (*callback)(void)) {
   ldv_handle = handle;
   callback_read = callback;
   /* Check for any posted input messages. Perform the callback if so. */
   if(pxi_tail->busy)
	callback_process();
}

/*
** This is the IOCTL IN function for the device. The API calls this via DOS
** in order to grab function pointers to the READ, WRITE, and our own
** REGISTER function. It then calls these functions directly rather
** than through the DOS device path.
*/
int ldv_ioctl_fn(struct adapter_info_s far *ap) {
    if(ldv_open) {
	ap->read_fn = MK_FP(_CS, FP_OFF(ldv_read_direct));
	ap->write_fn = MK_FP(_CS, FP_OFF(ldv_write_direct));
	ap->register_fn = MK_FP(_CS, FP_OFF(ldv_register_fn));
	ap->ioctl_stat = LDV_OK;
    } else
	ap->ioctl_stat = LDV_NOT_OPEN;
    return(LDV_OK);

}

/****************************************************************/
/* DRIVER INITIALIZATION FUNCTIONS                              */
/****************************************************************/

/*
char hail[] = "\r\n\
 Echelon (R) LON Parallel I/O MIP Driver Version 0.03\r\n\
 Copyright (C) Echelon Corporation 1992, 1993. All rights reserved.\r\n";
*/

char hail[] = "\r\n\
 LM104 MIP Driver Version 2.10";

char hail1[] =
#ifdef UPLINT
#ifdef DMAENA
" P50 Interrupt/DMA Interface \r\n";
#else
" P50 Interrupt I/O Interface\r\n";
#endif
#else
" P50 Polled I/O Interface\r\n";
#endif

char hail2[] = " Copyright (C) Vista Electronics, Inc. 1997. All rights reserved. \r\n";
/*
** DOS INT 21,02 : console print character.
*/
void dosputc(char cc) {
    asm mov	dl, cc
    asm mov	ah, 0x02
    asm int	0x21
}

/*
** Perform a series of character print calls on a NULL term'd string.
*/
void printd(char far *sp) {
    while(*sp)
	dosputc(*sp++);
}

/*
** Simple ascii to int for command line parsing.
** Return the new string pointer.
*/
unsigned i_nn;
char far *our_atoi(char far *sp) {
    char cc;

    i_nn = 0;
    while((cc = *sp) != 0) {
	if(cc >= '0' && cc <= '9') {
	    i_nn *= 10;
	    i_nn += (cc - '0');
	} else
	    return(sp);
	sp++;
    }
    return(sp);
}



/*
** The initialization function will parse arguments from the command
** line and grab the required memory from the heap.
** <cmdp> points to just after the '=' character in the command line.
** The command line form is:
** device=lm104mip.sys [switches]
** Where	/Onnn describes the output buffer count, nnn
** 		/Innn describes the input buffer count, nnn
**		/Pn describes an I/O port
**		/Dn decsribes the device id, n, for LON[n]
**		/W enables read token wait (polled I/O only)
**		/Z inhibits auto flush cancel.
**		/Qnn irQ level.
**
** Command lines are terminated with the '\r' character.
**
** Returns status and stuffs the end address in the request header.
*/
int ldv_init_fn(struct rh0_s far *hp) {

    int ii;
    BOOLEAN cmd_err = FALSE;
    char far *cmdp;
#ifndef UPLINT
    void interrupt (*vect_temp)(void);
#endif

    printd(hail);
    printd(hail1);
    printd(hail2);

    /* default: zero length driver if error: */
    hp->brk_address = NULL;

    // Defaults:
#ifdef UPLINT
#ifdef DMAENA
    dma_channel = 0;		// DMA channel 0
#endif
    mip_intrq = 5;		// Default was PC/AT IRQ 10
#endif

    /* Get the command line text: */
    cmdp = hp->cml_address;

    /* Skip the device driver name. */
    while(*cmdp > ' ')
	cmdp++;

    while(*cmdp >= ' ') {
	while(*cmdp >= ' ' && *cmdp != '/')
	    cmdp++;
	if(*cmdp == '/') {
	    ii = *++cmdp;
	    switch(ii) {
	    case 'D':
		/* Set the device number: "LON[n]" */
		cmdp = our_atoi(++cmdp);
		if(i_nn > 0 && i_nn < 10)
		    Header.dr_name[3] = i_nn + '0';
		else
		    cmd_err = TRUE;
		break;
	    case 'I':
		// Input buffer count option
		cmdp = our_atoi(++cmdp);
		if(i_nn > 1)
		    pxi_count = min(i_nn, MAX_BUFC);
		else
		    cmd_err = TRUE;
		break;
	    case 'O':
		// Output buffer count option
		cmdp = our_atoi(++cmdp);
		if(i_nn > 1)
		    pxo_count = min(i_nn, MAX_BUFC);
		else
		    cmd_err = TRUE;
		break;
	    case 'P':
		// I/O Port number option
		// modified to allow 4 options
		cmdp = our_atoi(++cmdp);
		if(i_nn > 0 && i_nn < 5)
		    switch(i_nn){
			case 1:
			    io_port_num = PPORT1;
			    break;
			case 2:
			    io_port_num = PPORT2;
			    break;
			case 3:
			    io_port_num = PPORT3;
			    break;
			case 4:
			    io_port_num = PPORT4;
			    break;
		    }	
		else
		    cmd_err = TRUE;
		break;

	    #ifdef UPLINT
	    case 'Q':
		/* Irq level select */
		cmdp = our_atoi(++cmdp);
		if(i_nn > 2 && i_nn < 15)
		    mip_intrq = i_nn;
		else
		    cmd_err = TRUE;
		break;
            #endif
	    case 'Z':
		/* disable auto flush cancel */
		g_afc_disable = TRUE;
		break;
	    default:
		printd(" *** Error, Unknown Switch :");
		dosputc(ii);
		printd("\007\r\n");

		/* skip white space: */
		while(*cmdp && *cmdp == ' ')
		    cmdp++;
		break;
	    }
	}else
	    break;
	if(cmd_err)
	    break;
    }
    printd("\r\n");

    hp->brk_address = drv_buffers_init() + 16;	/* Safety padding 16. */

    if(cmd_err) {
	printd(" *** Error in argument variable ***\007\r\n");
	return(LDV_NOT_OPEN);
    }
    pio_init();
#ifndef UPLINT
    // If all is well then install the DOS_TICK hook.
    vector_was = getvect(DOS_TICK);
    // Install our function in the chain:
    vect_temp = MK_FP(_CS, FP_OFF(tick_int_main));
    setvect(DOS_TICK, vect_temp);
#endif
    return(LDV_OK);
}

#ifdef DOSEXE
#include "MIP_MAIN.C"
#endif
