/*****************************************************************
    Filename: MIP_PIO.C
    Copyright 1992, 1993 (C) Echelon Corp. All rights reserved
    %W% %G% %U%
    Modified 10/10/94 By Don M. Scofield and Birkir Brimdal
		      for LM_104
    Modified 10/16/94 MHS for DMS
    Modified 11/01/94 DMS for DMS
	added delay for hardware
    Modified 12/09/94 RW and BB
	fix HS, remove debug and delay statements
    Modified 04/03/97 DMS and BB
	corrected reset_mip() and read_isr()
*****************************************************************/


/*
** This file contains the low level interface functions to the NEURON MIP
** via the user chosen parallel port.
**
*/

/* The segment, class, and group names prescribed in this module
** cause this entire module to be in the same physical segment */

/* This sets the data and BSS class to 'CODE': */
#pragma option -zTCODE -zBCODE

/* This sets the code, BSS, and data group to 'DGROUP': */
#pragma option -zPDGROUP -zSDGROUP -zGDGROUP

#include <dos.h>
#include "mip_typs.h"
#include "mdv_time.h"


/**********************************************************
**      PIO Interface I/O functions
**********************************************************/

/*
** This function is used to perform any sort of initialization
** required by the user's final parallel interface implementation.
** It is called when the driver is first loaded/initialized.
*/
void pio_init(void)
{
	reset_mip();
}

/*
** Hardware dink the reset line and clear out the ISR flags:
*/
void reset_mip(void) {
    int reset_loop;

    // this loop is required (due to the narrow IOW cycle width)
    // to insure the reset pin is held low long enough for a
    // proper neuron hardware reset.
    for (reset_loop=0; reset_loop<50; reset_loop++)
    {
	outportb(io_port_num+1, WCNTL_RESET);
    }
    // the delay is required to insure that the neuron reset line
    // has returned to the high state before clearing the flags.
    microsecond_delay(5000);
    outportb(io_port_num+1, WCNTL_CLR_FLGS);
}

/*
** This function reads the io_port_num+1 register. Bits are cleared on
** read.
*/
int read_isr(void) {
    int rtn_value;
    rtn_value = inportb(io_port_num+1);
    // clear out the ISR flags, not reset:
    outportb(io_port_num+1, WCNTL_CLR_FLGS);
    return(rtn_value);
}

/*
** This function test the MIP handshake bit only.
*/
int read_hs(void) {
    return(inportb(io_port_num+1) & RISR_HS);
}


/*
** This function waits for handshake completion from the NEURON,
** (HS = 0). It utilizes a timeout to prevent lockup.
** Returns FALSE if timed out. For speed purposes the HS line is sampled
** as an initial step. Timeout errors can be interpreted as a NEURON reset.
*/
BOOLEAN wait_for_hs(void) {
    static LdvTimer l_timer;

    if((inportb(io_port_num+1) & RISR_HS) == 0)
	return(TRUE);

    // Reset test if not HS:
    if(mip_was_reset)
	return(FALSE);

    // Timeout on HS is 400ms. */
    timer_start(400, &l_timer);
    do {
	if((inportb(io_port_num+1) & RISR_HS) == 0)
	    return(TRUE);
    } while(!timeout_check(&l_timer));
    mip_was_reset = TRUE;       // This aborts most interface ops.
    return(FALSE);
}


/*
** Just return a zero if there was a problem.
*/
byte pio_read(void) {

    if(inportb(io_port_num+1) & RISR_HS) {
	if(!wait_for_hs())
	    return(0);
    }
    return(inportb(io_port_num));
}

/*
** Single byte data write.
*/
void pio_write(byte pdata) {

    if(inportb(io_port_num+1) & RISR_HS) {
	if(!wait_for_hs())
	    return;
    }
    outportb(io_port_num, pdata);
}
